// #define NUM_USER_TEXCOORDS 1

#include "CommonParams.h"
#include "UserShader.fx"
#include "Light.fx"

#ifdef SHADOWMAP
#include "ShadowMap.h"
#endif

struct vertexInput
{
    float4 Position : POSITION0;
    float3 Normal   : NORMAL0;
    float2 Height   : TEXCOORD0;
#if NUM_USER_TEXCOORDS
    float2 TexCoords[NUM_USER_TEXCOORDS] : TEXCOORD1;
#endif
};

struct vertexOutput
{
    float4 outPosition      : POSITION;
    float4 WorldPosition    : TEXCOORD0;
    float4 Normal           : TEXCOORD1; // Normal.w ... fog
    float4 ScreenPosition   : TEXCOORD2;
#if NUM_USER_TEXCOORDS
    float2 TexCoords[NUM_USER_TEXCOORDS] : TEXCOORD3;
#endif
};

bool  bAni = true;
float fGrassTime = 0;

struct WindParams
{
    int   windState;
    bool  bEnableWave1;
    bool  bEnableWave2;
    int   nSway;
    float3 fDirectDelta;
    float2 fAmplitude;
    float2 fWaveLen;
    float2 fWindSpeed;
    float2 fWaveUp;
    float2 fWaveDown;
    float2 fOffset;
};

WindParams g_WindParams = 
{
    0,
    true,
    true,
    0,
    0.1f, 2.0f, 1.0f,
    0.5f, 0.8f,
    1000.0f, 500.0f,
    1.0f, 2.0f,
    1.0f, 1.0f,
    0.1f, 0.1f,
    0.0f, 0.0f
};

float SumOfDirectionSineWaves(float x, float z, float dirx, float dirz)
{
    float  d = sqrt(x * x + z * z);
    float3 f0 = normalize(float3(x, 0, x));
    float3 f1 = normalize(float3(dirx, 0, dirz));
    float  s = d * dot(f0, f1);
    
    float sum = 0.0f;
    float sintime = sin(fGrassTime);
    
    if (g_WindParams.bEnableWave1)
    {
        float f = g_WindParams.fAmplitude.x * sin(s / g_WindParams.fWaveLen.x - fGrassTime * g_WindParams.fWindSpeed.x);
        
        f = f + sintime * g_WindParams.fOffset.x;        
        if (f >= 0.0f)
            f = f * g_WindParams.fWaveUp.x;
        else
            f = f * g_WindParams.fWaveDown.x;
            
        sum += f;
    }
    
    if (g_WindParams.bEnableWave2)
    {
        float f = g_WindParams.fAmplitude.y * sin(s / g_WindParams.fWaveLen.y - fGrassTime * g_WindParams.fWindSpeed.y);
        
        f = f + sintime * g_WindParams.fOffset.y;        
        if (f >= 0.0f)
            f = f * g_WindParams.fWaveUp.y;
        else
            f = f * g_WindParams.fWaveDown.y;
            
        sum += f;
    }
    
    return sum * sintime * 20.0f;
}

float3 EffectPos(float3 Pos, float2 UV, float2 Height)
{
    if (g_WindParams.windState == 2)
        return Pos;
        
    float swayOffset = Height.x;
    float ta = fGrassTime + Pos * 0.322f;
    
    if (g_WindParams.nSway == 0)
        ta += UV.x;
    else if (g_WindParams.nSway == 1)
        ta += Height.x;
    else
        ta = fGrassTime;
    
    if (g_WindParams.windState == 1)
    {
        float fDelta = SumOfDirectionSineWaves(Pos.x, Pos.z, g_WindParams.fDirectDelta.x, g_WindParams.fDirectDelta.z) * swayOffset;
        Pos += 10.0f * sin(ta) * swayOffset;
        Pos += fDelta * g_WindParams.fDirectDelta * swayOffset;
    }
    else if (g_WindParams.windState == 0)
    {
        Pos += 10.0f * sin(ta) * swayOffset;
    }
    return Pos;
}

MaterialPixelParameters GetMaterialParameters(vertexOutput Input)
{
    MaterialPixelParameters Params;
#if NUM_USER_TEXCOORDS
    [unroll(NUM_USER_TEXCOORDS)]for (int i = 0; i < NUM_USER_TEXCOORDS; i++)
        Params.TexCoords[i] = Input.TexCoords[i];
#endif
    Params.WorldNormal = Input.Normal.xyz;//.w fog
    return Params;
}

vertexOutput VertexShaderMain(vertexInput Input)
{
    vertexOutput Output = (vertexOutput)0;

	float3 vPos = Input.Position.xyz;


#if NUM_USER_TEXCOORDS
    [unroll(NUM_USER_TEXCOORDS)]for (int i = 0; i < NUM_USER_TEXCOORDS; i++)
    Output.TexCoords[i] = Input.TexCoords[i];
    
    if (bAni)
	    vPos = EffectPos(vPos, Input.TexCoords[0], Input.Height);
#endif	    
    Output.WorldPosition  = mul(float4(vPos, 1.0f), world);
    Output.ScreenPosition = mul(float4(vPos, 1.0f), wvp);
    Output.outPosition = Output.ScreenPosition;
    
    float3 normal = Input.Normal;
    if(!any(normal))
        normal=float3(0,1,0);
    Output.Normal.xyz = normalize(mul(normal, world));
    Output.Normal.w   = CalFogFactor(Output.outPosition.z);
    return Output;
}

float4 PixelShaderMain(vertexOutput Input 
                    #if defined (SHADOWMAP) || defined(COVERPOINTLIGHT) || defined(POINTLIGHT)
                       ,float4 screenSpace : VPOS
                    #endif    
                       ) : COLOR0
{
    float4 retColor = 0;
    MaterialPixelParameters Params = GetMaterialParameters(Input);
    CalcMaterialPixelParameters(Params,Input.ScreenPosition,Input.WorldPosition.xyz);
    
	//ֹģ ⲿظģҪҪᵽmaterialtemplateȥ
#ifdef LM_UNLIGHT
	retColor.rgb = GetEmissiveColor(Params);
	retColor.a = GetOpacityColor(Params);
#endif
#ifdef LM_LAMBERT
	float3 ambientColor = GetAmbientColor(Params);
	float3 emissiveColor = GetEmissiveColor(Params);
	float3 diffuseColor = GetDiffuseColor(Params);
#ifdef SHADOWMAP
	float shadowMask = 1.0f;
	if (shadowMapParam.bShadowMask)
	{
        		float2 screenpos = GetScreenpos(screenSpace.xy);
        		shadowMask = GetShadowMask(screenpos);
	}
#endif  
	retColor.rgb = computeLambert(
		ambientColor,
		emissiveColor,
		diffuseColor,
		Params.WorldNormal
#if defined(POINTLIGHT) || defined(COVERPOINTLIGHT)
		,screenSpace
#endif
#ifdef SHADOWMAP
		, shadowMask
#endif
		);
	retColor.a = GetOpacityColor(Params);
#endif
#ifdef LM_PHONG
	float3 ambientColor = GetAmbientColor(Params);
	float3 emissiveColor = GetEmissiveColor(Params);
	float3 diffuseColor = GetDiffuseColor(Params);
	float3 specularColor = GetSpecularColor(Params);
	float specularPower = GetSpecularPower(Params);
	if(specularPower<0.00001)
		specularPower = 15;
	float3 V = normalize(eyes - Params.WorldPosition);
#ifdef SHADOWMAP
	float shadowMask = 1.0f;
	if (shadowMapParam.bShadowMask)
	{
        		float2 screenpos = GetScreenpos(screenSpace.xy);
        		shadowMask = GetShadowMask(screenpos);
	}
#endif  
	retColor.rgb = computePhong(
		ambientColor,
		emissiveColor,
		diffuseColor,
		specularColor,
		specularPower,
		Params.WorldNormal
#if defined(POINTLIGHT) || defined(COVERPOINTLIGHT)
		,screenSpace
#endif
		,V
#ifdef SHADOWMAP
		,shadowMask
#endif
		);
	retColor.a = GetOpacityColor(Params);
#endif
#ifdef LM_ANISOTROPIC

#endif
#ifdef LM_SSS

#endif
#ifdef LM_CUSTOM//CUSTOMLIGHTING
    retColor = GetCustomLighting(Params);
#endif
#ifdef LM_JX3
    float3 color = 0;
    color = GetDiffuseColor(Params);
#ifdef SHADOWMAP
	float shadowMask = 1.0f;
    if (shadowMapParam.bShadowMask)
    {
        float2 screenpos = GetScreenpos(screenSpace.xy);
        shadowMask = GetShadowMask(screenpos);
    }
#endif  
#ifdef SPECULAR
	float3 specularColor = GetSpecularColor(Params);
	float specularPower = GetSpecularPower(Params);
	if(specularPower < 0.0001)
		specularPower = 15;
#endif
    float3 light = ComputeLight(
					Params
#ifdef SPECULAR
					,specularColor,
					specularPower,
					0.5
#endif
#ifdef SSS
					,0.1
#endif
#ifdef SHADOWMAP
					,shadowMask
#endif
#if defined(POINTLIGHT) || defined(COVERPOINTLIGHT)
		,screenSpace
 #endif
					);   //use worldnormal inside
    color *= light;
    color +=  GetAmbientColor(Params);
    color +=  GetEmissiveColor(Params);
    retColor.rgb = color;
    retColor.a = GetOpacityColor(Params);
#endif    //light mode     
#ifdef FOG
    retColor = CalFogColor(retColor, Input.Normal.w);
#endif
    return retColor;
}

technique Color
{
    pass p0
    {
        VertexShader = compile vs_3_0 VertexShaderMain(); 
		PixelShader = compile ps_3_0 PixelShaderMain();
	
        CullMode = NONE; 
        ZEnable = true; 
        ZWriteEnable = true;
        AlphaTestEnable = true;
        AlphaBlendEnable = false;
        AlphaFunc = GreaterEqual;
        AlphaRef  = 0x80;
    }
}

technique ColorSoftMask
{
	pass p0
	{
		VertexShader = compile vs_3_0 VertexShaderMain(); 
		PixelShader = compile ps_3_0 PixelShaderMain();
		AlphaTestEnable = true;
		AlphaBlendEnable = false;
		ZEnable = true;
		ZWriteEnable = true;
		ZFunc = LESSEQUAL;
		CullMode = cw; 
	}
	pass p1
	{
		VertexShader = compile vs_3_0 VertexShaderMain(); 
	    PixelShader = compile ps_3_0 PixelShaderMain();
		AlphaTestEnable = false;
		AlphaBlendEnable = true;
		ZEnable = true;
		ZWriteEnable = false;
		ZFunc = LESS;
		CullMode = cw; 
	}
}

struct ZOutput
{
    float4    ScreenPos : POSITION;
    float2    Texcoord  : TEXCOORD0;
    float2    RawPos    : TEXCOORD1;
};

ZOutput VS_EarlyZ(vertexInput Input) 
{
    ZOutput Out;
    Out.ScreenPos = mul(Input.Position, wvp);
    Out.RawPos.x = Out.ScreenPos.z;
    Out.RawPos.y = Out.ScreenPos.w;
    Out.Texcoord = Input.TexCoords[0];
    return Out;
}

float4 PS_EarlyZ(ZOutput Input) : COLOR
{
    MaterialPixelParameters Params = (MaterialPixelParameters)0;
#if NUM_USER_TEXCOORDS
    [unroll(NUM_USER_TEXCOORDS)]for (int i = 0; i < NUM_USER_TEXCOORDS; i++)
        Params.TexCoords[i] = Input.Texcoord;
#endif
    float4 color = GetOpacityColor(Params);	
    color.rgb = PackingFloatToRGB(Input.RawPos.x / Input.RawPos.y).rgb;
    return color;
}

technique EARLYZ
{
    pass p0
    {
        VertexShader = compile vs_3_0 VS_EarlyZ(); 
        PixelShader = compile ps_3_0 PS_EarlyZ();
        
        AlphaBlendEnable = false; 
        CullMode = cw; 
        ZWriteEnable = true;
		FogEnable = false;
    }
}